/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/

/* ---------------------------------------------------------------
 * File: xperfseq.c 
 * performance sequencer functions                                  
 * -----------------------------------------------------------------*/

#include <xpciapi.h>
#include <xregx20.h>
#include <xregx21.h>
#include <xaddrmap.h>

/* Pointer to performance sequencer database */
#define DB_PERFSEQ(measure) (bx_handlearray[handle].db->Analyzer.PerfSeq[measure])

/* Value of generic property */
#define GENERIC(measure,prop) (DB_PERFSEQ(measure).GenProp[prop])

/* Value of tran property */
#define TRAN(measure,tran,prop) (DB_PERFSEQ(measure).Line[tran].Transient[prop])

/* Value of cond property */
#define COND(measure,tran,prop) (DB_PERFSEQ(measure).Line[tran].Condition[prop])


/****************************************************************************
  Generic properties ********************************************************
****************************************************************************/

/*---------------------------------------------------------------------------*
 * BestXPerfGenSet
 *
 * Purpose: Sets a generic property for a performance measure
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXPerfGenSet(
  bx_handletype handle,
  bx_int32 measure,
  bx_perfgentype prop,
  bx_int32 val
)
{
  BX_DECLARE_FUNCNAME ("BestXPerfGenSet [perfgenset]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* 
       No performance licensecheck here,
       because DBInit() must always succeed
       (as long as CAPI license available).
    */
    
    BX_LICENSECHECK(BX_CAPABILITY_CAPI);
  
    /* range checking */
    BX_FCT_PARAM_CHK(1,measure>=BX_PERFSEQ_SIZE);
    BX_TRY(BestXParamCheck(handle,BX_PARAM_PERFSEQ_GEN,(bx_int32)prop,val));
    GENERIC(measure,prop)=val;
  }
  BX_ERRETURN(BX_TRY_RET);
}


/*---------------------------------------------------------------------------*
 * BestXPerfGenGet
 *
 * Purpose: Gets a generic property of a performance measure
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXPerfGenGet(bx_handletype handle,
  bx_int32 measure,
  bx_perfgentype prop,
  bx_int32 * val)
{
  BX_DECLARE_FUNCNAME ("BestXPerfGenGet [perfgenget]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* 
       No performance licensecheck here,
       because DBInit() must always succeed
       (as long as CAPI license available).
    */
    
    BX_LICENSECHECK(BX_CAPABILITY_CAPI);

    /* range checking */
    BX_FCT_PARAM_CHK(1,measure>=BX_PERFSEQ_SIZE);
    BX_TRY(BestXParamProbe(handle,BX_PARAM_PERFSEQ_GEN,(bx_int32)prop));
    BX_TRY_FCT_PARAM_NULL_POINTER(val);
    
    *val=GENERIC(measure,prop);
  }
  BX_ERRETURN(BX_TRY_RET);
}


/*---------------------------------------------------------------------------*
 * BestXPerfGenDefaultSet()
 *
 * Purpose: Sets all generic props for a measure to default
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXPerfGenDefaultSet(
  bx_handletype handle,
  bx_int32 measure
)
{
  BX_DECLARE_FUNCNAME("BestXPerfGenDefaultSet [perfgendefset]");

  BX_TRY_VARS_NO_PROG;
  bx_int32 i;

  /* Generic info for range checking */
  const bx_generic_infotype *GenInfo;
  const bx_param_infotype *ParamInfo;

  BX_TRY_BEGIN
  {
    /* 
       No performance licensecheck here,
       because DBInit() must always succeed
       (as long as CAPI license available).
    */
    
    BX_LICENSECHECK(BX_CAPABILITY_CAPI);
    
    BX_FCT_PARAM_CHK(1,measure>=BX_PERFSEQ_SIZE);

    /* Get pointer to generic info */
    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_PERFSEQ_GEN, &GenInfo));

    for (i = 0; i < GenInfo->num_elem; i++)
    {
      /* Get pointer to i-th (existing) property */
      BX_TRY(BestXParamInfoGet(handle, BX_PARAM_PERFSEQ_GEN,
          i, &ParamInfo, (bx_int32)BX_INDEX_SEARCH));

      /* Set it to default */
      BX_TRY(BestXPerfGenSet(handle,measure,
            ParamInfo->proptyp.perfseqgenprop,
            ParamInfo->defaultval));
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/****************************************************************************
  TRAN properties ********************************************************
****************************************************************************/

/*---------------------------------------------------------------------------*
 * BestXPerfTranSet
 *
 * Purpose: Sets a tran property for a performance measure
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXPerfTranSet(
  bx_handletype handle,
  bx_int32 measure,
  bx_int32 transient,
  bx_perftrantype prop,
  bx_int32 val
)
{
  BX_DECLARE_FUNCNAME ("BestXPerfTranSet");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* 
       No performance licensecheck here,
       because DBInit() must always succeed
       (as long as CAPI license available).
    */
    
    BX_LICENSECHECK(BX_CAPABILITY_CAPI);
    
    /* range checking */
    BX_FCT_PARAM_CHK(1,measure>=BX_PERFSEQ_SIZE); 
    BX_FCT_PARAM_CHK(2,transient>=BX_PERFLINE_SIZE); 
    BX_TRY(BestXParamCheck(handle,BX_PARAM_PERFSEQ_TRAN,(bx_int32)prop,val));
    TRAN(measure,transient,prop)=val;
  
    /* Track the number of lines with maximum-function.
       Caution: 
       transient is zero-based, MaxNumLines is one-based ! 
    */
    if (transient+1 > DB_PERFSEQ(measure).MaxNumLines)
    {
      DB_PERFSEQ(measure).MaxNumLines=transient+1;
    }
  }
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXPerfTranGet
 *
 * Purpose: Gets a tran property for a performance measure
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXPerfTranGet(
  bx_handletype handle,
  bx_int32 measure,
  bx_int32 transient,
  bx_perftrantype prop,
  bx_int32 *val
)
{
  BX_DECLARE_FUNCNAME ("BestXPerfTranGet");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* range checking */
    BX_FCT_PARAM_CHK(1,measure>=BX_PERFSEQ_SIZE); 
    BX_FCT_PARAM_CHK(2,transient>=BX_PERFLINE_SIZE); 
    BX_TRY(BestXParamProbe(handle,BX_PARAM_PERFSEQ_TRAN,(bx_int32)prop));
    BX_TRY_FCT_PARAM_NULL_POINTER(val);

    *val=TRAN(measure,transient,prop);
  }
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXPerfTranDefaultSet()
 *
 * Purpose: Sets all tran props for a measure and transient to default
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXPerfTranDefaultSet(
  bx_handletype handle,
  bx_int32 measure,
  bx_int32 transient
)
{
  BX_DECLARE_FUNCNAME("BestXPerfTranDefaultSet []");

  BX_TRY_VARS_NO_PROG;
  bx_int32 i;

  /* Generic info for range checking */
  const bx_generic_infotype *GenInfo;
  const bx_param_infotype *ParamInfo;

  BX_TRY_BEGIN
  {
    /* license checking */
    /* 
       No performance licensecheck here,
       because DBInit() must always succeed
       (as long as CAPI license available).
    */
    
    BX_LICENSECHECK(BX_CAPABILITY_CAPI);
    
    BX_FCT_PARAM_CHK(1,measure>=BX_PERFSEQ_SIZE);
    BX_FCT_PARAM_CHK(2,transient>=BX_PERFLINE_SIZE); 

    /* Get pointer to generic info */
    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_PERFSEQ_TRAN, &GenInfo));

    for (i = 0; i < GenInfo->num_elem; i++)
    {
      /* Get pointer to i-th (existing) property */
      BX_TRY(BestXParamInfoGet(handle, BX_PARAM_PERFSEQ_TRAN,
          i, &ParamInfo, (bx_int32)BX_INDEX_SEARCH));

      /* Set it to default */
      BX_TRY(BestXPerfTranSet(handle,measure,transient,
            ParamInfo->proptyp.perfseqtranprop,
            ParamInfo->defaultval));
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/****************************************************************************
  COND properties ********************************************************
****************************************************************************/

/*---------------------------------------------------------------------------*
 * BestXPerfCondSet
 *
 * Purpose: Sets a cond property for a performance measure
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXPerfCondSet(
  bx_handletype handle,
  bx_int32 measure,
  bx_int32 transient,
  bx_perfcondtype prop,
  bx_charptrtype cond
)
{
  BX_DECLARE_FUNCNAME ("BestXPerfCondSet");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* 
       No performance licensecheck here,
       because DBInit() must always succeed
       (as long as CAPI license available).
    */
    
    BX_LICENSECHECK(BX_CAPABILITY_CAPI);
    
    /* range checking */
    BX_FCT_PARAM_CHK(1,measure>=BX_PERFSEQ_SIZE); 
    BX_FCT_PARAM_CHK(2,transient>=BX_PERFLINE_SIZE); 
    BX_TRY(BestXParamProbe(handle,BX_PARAM_PERFSEQ_COND,(bx_int32)prop));
    BX_TRY_FCT_PARAM_NULL_POINTER(cond);
    
    if (COND(measure,transient,prop))
    {
      /* first free the old string */
      BestXMemFree((void **) &(COND(measure,transient,prop))); 
      COND(measure,transient,prop)=NULL;
    }
    
    /* Allocate the new string */
    COND(measure,transient,prop)=(bx_charptrtype) BestXMemCalloc(BESTX_STRLEN(cond)+1,sizeof(bx_int8));
    BX_TRY_FAIL((COND(measure,transient,prop))?BX_E_OK:BX_E_HOST_MEM_FULL);
    
    /* Set property value */
    BESTX_STRCPY(COND(measure,transient,prop),cond);

    /* Track the number of lines with maximum-function.
       Caution: 
       transient is zero-based, MaxNumLines is one-based ! 
    */
    if (transient+1 > DB_PERFSEQ(measure).MaxNumLines)
    {
      DB_PERFSEQ(measure).MaxNumLines=transient+1;
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXPerfCondGet
 *
 * Purpose: Gets a cond property for a performance measure
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXPerfCondGet(
  bx_handletype handle,
  bx_int32 measure,
  bx_int32 transient,
  bx_perfcondtype prop,
  bx_charptrtype *cond
)
{
  BX_DECLARE_FUNCNAME ("BestXPerfCondGet");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* range checking */
    BX_FCT_PARAM_CHK(1,measure>=BX_PERFSEQ_SIZE); 
    BX_FCT_PARAM_CHK(2,transient>=BX_PERFLINE_SIZE); 
    BX_TRY(BestXParamProbe(handle,BX_PARAM_PERFSEQ_COND,(bx_int32)prop));
    BX_TRY_FCT_PARAM_NULL_POINTER(cond);

    *cond=COND(measure,transient,prop);
  }
  BX_ERRETURN(BX_TRY_RET);
}


/*---------------------------------------------------------------------------*
 * BestXPerfCondDefaultSet()
 *
 * Purpose: Sets all cond props for a measure and transient to default
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXPerfCondDefaultSet(
  bx_handletype handle,
  bx_int32 measure,
  bx_int32 transient
)
{
  BX_DECLARE_FUNCNAME("BestXPerfCondDefaultSet []");

  BX_TRY_VARS_NO_PROG;
  bx_int32 i;

  /* Generic info for range checking */
  const bx_generic_infotype *GenInfo;
  const bx_param_infotype *ParamInfo;

  BX_TRY_BEGIN
  {
    /* license checking */
    /* 
       No performance licensecheck here,
       because DBInit() must always succeed
       (as long as CAPI license available).
    */
    BX_LICENSECHECK(BX_CAPABILITY_CAPI);

    BX_FCT_PARAM_CHK(1,measure>=BX_PERFSEQ_SIZE);
    BX_FCT_PARAM_CHK(2,transient>=BX_PERFLINE_SIZE); 

    /* Get pointer to generic info */
    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_PERFSEQ_COND, &GenInfo));

    for (i = 0; i < GenInfo->num_elem; i++)
    {
      /* Get pointer to i-th (existing) property */
      BX_TRY(BestXParamInfoGet(handle, BX_PARAM_PERFSEQ_COND,
          i, &ParamInfo, (bx_int32)BX_INDEX_SEARCH));

      /* Set it to default */
      BX_TRY(BestXPerfCondSet(handle,measure,transient,
            ParamInfo->proptyp.perfseqtranprop,
            ParamInfo->defaultstr));
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/****************************************************************************
  TRAN + COND properties ****************************************************
****************************************************************************/

/*---------------------------------------------------------------------------*
 * BestXPerfTranCondDefaultSet()
 *
 * Purpose: Sets all trans and cond props for a measure and transient to default
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXPerfTranCondDefaultSet(
  bx_handletype handle,
  bx_int32 measure,                              
  bx_int32 transient         
)
{
  BX_DECLARE_FUNCNAME("BestXPerfTranCondDefaultSet []");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* 
       No performance licensecheck here,
       because DBInit() must always succeed
       (as long as CAPI license available).
    */
    
    BX_LICENSECHECK(BX_CAPABILITY_CAPI);

    /* Range checking */
    BX_FCT_PARAM_CHK(1,measure>=BX_PERFSEQ_SIZE);
    BX_FCT_PARAM_CHK(2,transient>=BX_PERFLINE_SIZE); 

    BX_TRY(BestXPerfTranDefaultSet(handle,measure,transient));
    BX_TRY(BestXPerfCondDefaultSet(handle,measure,transient));
  }

  BX_ERRETURN(BX_TRY_RET);
}


/*---------------------------------------------------------------------------*
 * BestXPerfDefaultSet()
 *
 * Purpose: Initialises the descriptor table of the sequencer               
 *          of the selected measure                                  
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXPerfDefaultSet(
  bx_handletype handle,
  bx_int32 measure
)
{
  BX_DECLARE_FUNCNAME("BestXPerfDefaultSet []");

  bx_int32 t; /* transient */

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* 
       No performance licensecheck here,
       because DBInit() must always succeed
       (as long as CAPI license available).
    */
    
    BX_LICENSECHECK(BX_CAPABILITY_CAPI);

    /* Range checking */
    BX_FCT_PARAM_CHK(1,measure>=BX_PERFSEQ_SIZE);
    
    for (t=0;t<BX_PERFLINE_SIZE;t++)
    {
      BX_TRY(BestXPerfTranCondDefaultSet(handle,measure,t));
    }

    /* Here we have the same default line 128 times.
       To avoid programming all lines we reduce
       MaxNumLines to one.
    */

    assert(DB_PERFSEQ(measure).MaxNumLines==BX_PERFLINE_SIZE); /* both one-based ! */
    DB_PERFSEQ(measure).MaxNumLines=1;
  }

  BX_ERRETURN(BX_TRY_RET);
}


/****************************************************************************
  Actions *******************************************************************
****************************************************************************/

/*---------------------------------------------------------------------------*
 * BestXPerfCtrRead()
 *
 * Purpose: Returns the contents of the performance counter registers                                
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXPerfCtrRead(
  bx_handletype handle,
  bx_int32 measure,
  bx_int32 counter,
  bx_int32 * val)
{
  BX_DECLARE_FUNCNAME ("BestXPerfCtrRead");

  bx_int8 in_zw[IN_PERF_CTR_READ];
  bx_int8 out_zw[OUT_PERF_CTR_READ];
  bx_int16 addr;
  bx_int8ptr bp;
  bx_int32 mephisto_version;

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    BX_LICENSECHECK(BX_CAPABILITY_PERFORMANCE|BX_CAPABILITY_CAPI);
    
    /* Range checking */
    /* first check the current board version as of bytecount bugfix ^2*/
    BX_TRY(BestXDirectRegRead(handle, BX_REG_CHIP_REV_REG,2,&mephisto_version));
    
    if(mephisto_version == BX_MEPHISTO_MMC1_MMC2)
    { 
      BX_FCT_PARAM_CHK(1,measure>=BX_PERFSEQ_SIZE_MMC1_MMC2);
    }
    else
    {
      BX_FCT_PARAM_CHK(1,measure>=BX_PERFSEQ_SIZE);
    }

    BX_FCT_PARAM_CHK(2,counter>=BX_PERFCTR_SIZE);
    BX_TRY_FCT_PARAM_NULL_POINTER(val);

    /* Put measure and counter into the bytestream */
    bp = BestXLong2Stream(in_zw, &measure, 1UL); 
    bp = BestXLong2Stream(bp, &counter, 1UL); 

    addr = OUT_PERF_CTR_READ;
    if(BestXHasFirmware(handle))
    {
      BX_TRY(BestXBasicCommand(handle, CMD_PERF_CTR_READ,in_zw, IN_PERF_CTR_READ, out_zw, &addr));
    }
    (void) BestXStream2Long(val, out_zw, 1UL);
  }

  BX_ERRETURN(BX_TRY_RET);
}


/*---------------------------------------------------------------------------*
 * BestXPerfProg()
 *
 * Purpose: Programs the  sequencer memory of the performance counter       
 *          of the selected measure                             
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXPerfProg(
  bx_handletype handle,
  bx_int32 measure
)
{
  bx_int32 val,offset;
  bx_int32 mephisto_version;

  BX_DECLARE_FUNCNAME("BestXPerfProg [perfprog]");
  BX_TRY_VARS;
  
  BX_TRY_BEGIN
  {
    BX_LICENSECHECK(BX_CAPABILITY_PERFORMANCE|BX_CAPABILITY_CAPI);
    
    /* lock resource */
    BX_TRY_PROGRESS(BestXResourceLock(handle,BX_RESLOCK_PERFORMANCE));
    
    /* Range checking */
    /* first check the current board version as of bytecount bugfix ^2*/
    BX_TRY(BestXDirectRegRead(handle, BX_REG_CHIP_REV_REG,2,&mephisto_version));
    
    if(mephisto_version == BX_MEPHISTO_MMC1_MMC2)
    { 
      BX_FCT_PARAM_CHK(1,measure>=BX_PERFSEQ_SIZE_MMC1_MMC2);
    }
    else
    {
      BX_FCT_PARAM_CHK(1,measure>=BX_PERFSEQ_SIZE);
    }


    /* set generic properties in hardware */

    /* CAUTION: Due to the performance ctr bugfix in BestXSequencerProg
       we have to set the generic properties first */
    
    offset = BX_REG_BLK1_PERF_DBI_REG_0 - BX_REG_BLK0_PERF_DBI_REG_0;  
    BX_TRY(BestXPerfGenGet(handle,measure,BX_PERFGEN_CTRAMODE,&val));
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_BLK0_BC_EN_REG + (offset * measure),sizeof(bx_int16),val));
    BX_TRY(BestXPerfGenGet(handle,measure,BX_PERFGEN_FBA,&val));
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_BLK0_PERF_PL_REG + (offset * measure),sizeof(bx_int32),val));

    BX_TRY(BestXSequencerProg(handle,(bx_seqtype)(BX_SEQ_PERF_0+measure)));

    /* unlock resource */
    BX_TRY_PROGRESS(BestXResourceUnlock(handle,BX_RESLOCK_PERFORMANCE));
  }

  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Resource locking OK */
      BX_TRY_PASSED
      {
        /* Resource unlocking OK */
      }
      BX_TRY_FAILED
      {
        /* some function returned an error */
        (void)BestXResourceUnlock(handle, BX_RESLOCK_PERFORMANCE);
      }
    }
    BX_TRY_FAILED
    {
      /* Resource locking failed */
    }
  }



  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXPerfRun()
 *
 * Purpose: Starts all performance counters                           
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXPerfRun(bx_handletype handle)
{
  BX_DECLARE_FUNCNAME ("BestXPerfRun");
  BX_TRY_VARS;
  
  BX_TRY_BEGIN
  {
    BX_LICENSECHECK(BX_CAPABILITY_PERFORMANCE|BX_CAPABILITY_CAPI);
    
    /* lock resource. */
    BX_TRY_PROGRESS(BestXResourceLock(handle,BX_RESLOCK_PERFORMANCE));

    /* Start watchtask and init counters in FW */
    if(BestXHasFirmware(handle))
    {
      BX_TRY(BestXBasicCommand(handle, CMD_PERF_RUN, NULL, IN_PERF_RUN,
      NULL, NULL));
    }
    /* Run performance */
    BX_TRY(BestXSyncRegWrite(handle,
                           
                           BX_REG_AN_EN_STATUS,
                           sizeof(bx_int16),
                           0x1,
                           0x1,
                           
                           BX_REG_AN_EN_REG,
                           sizeof(bx_int16),
                           0x1,
                           0x1,
                           
                           BX_REG_AN_EN_SREG,
  
                           BX_REG_AN_EN_FORCE_REG,
                           sizeof(bx_int16),
                           0 /* no soft force */
                           ));


    BX_TRY_PROGRESS(BestXResourceUnlock(handle, BX_RESLOCK_PERFORMANCE));
  }

  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Resource locking OK */
      BX_TRY_PASSED
      {
        /* Resource unlocking OK */
      }
      BX_TRY_FAILED
      {
        /* some function returned an error */
        (void)BestXResourceUnlock(handle, BX_RESLOCK_PERFORMANCE);
      }
    }
    BX_TRY_FAILED
    {
      /* Resource locking failed */
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}


/*---------------------------------------------------------------------------*
 * BestXPerfStop()
 *
 * Purpose: Stops all performance counters                           
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXPerfStop(bx_handletype handle)
{
  BX_DECLARE_FUNCNAME ("BestXPerfStop");
  BX_TRY_VARS;
  
  BX_TRY_BEGIN
  {
    BX_LICENSECHECK(BX_CAPABILITY_PERFORMANCE|BX_CAPABILITY_CAPI);

    /* lock resource. */
    BX_TRY_PROGRESS(BestXResourceLock(handle,BX_RESLOCK_PERFORMANCE));

    /* Stop watchtask */
    if(BestXHasFirmware(handle))
    {
      BX_TRY(BestXBasicCommand(handle, CMD_PERF_STOP, NULL, IN_PERF_STOP,
      NULL, NULL));
    }
    /* Stop performance */
    BX_TRY(BestXSyncRegWrite(handle,
                           
                           BX_REG_AN_EN_STATUS,
                           sizeof(bx_int16),
                           0x1,
                           0x1,
                           
                           BX_REG_AN_EN_REG,
                           sizeof(bx_int16),
                           0x1,
                           0x0,
                           
                           BX_REG_AN_EN_SREG,
  
                           BX_REG_AN_EN_FORCE_REG,
                           sizeof(bx_int16),
                           0 /* no soft force */
                           ));



    /* unlock resource. locking was done in BestXPerfRun !!!*/
    BX_TRY_PROGRESS(BestXResourceUnlock(handle,BX_RESLOCK_PERFORMANCE));
  }

  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Resource locking OK */
      BX_TRY_PASSED
      {
        /* Resource unlocking OK */
      }
      BX_TRY_FAILED
      {
        /* some function returned an error */
        (void)BestXResourceUnlock(handle, BX_RESLOCK_PERFORMANCE);
      }
    }
    BX_TRY_FAILED
    {
      /* Resource locking failed */
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}
/*---------------------------------------------------------------------------*
 * BestXPerfUpdate()
 *
 * Purpose: This call updates the performance counter registers and 
 *          restarts all counters.                        
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXPerfUpdate(bx_handletype handle)
{
  BX_DECLARE_FUNCNAME ("BestXPerfUpdate");
  BX_TRY_VARS;
  BX_TRY_BEGIN
  {
    BX_LICENSECHECK(BX_CAPABILITY_PERFORMANCE|BX_CAPABILITY_CAPI);
    
    /* lock resource. */
    BX_TRY_PROGRESS(BestXResourceLock(handle,BX_RESLOCK_PERFORMANCE));

    if(BestXHasFirmware(handle))
    {
      BX_TRY(BestXBasicCommand(handle, CMD_PERF_UPDATE, NULL, IN_PERF_UPDATE,
      NULL, NULL));
    }

    BX_TRY_PROGRESS(BestXResourceUnlock(handle, BX_RESLOCK_PERFORMANCE));
  }

  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Resource locking OK */
      BX_TRY_PASSED
      {
        /* Resource unlocking OK */
      }
      BX_TRY_FAILED
      {
        /* some function returned an error */
        (void)BestXResourceUnlock(handle, BX_RESLOCK_PERFORMANCE);
      }
    }
    BX_TRY_FAILED
    {
      /* Resource locking failed */
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/****************************************************************************
  Clock switching ***********************************************************
****************************************************************************/

/*---------------------------------------------------------------------------*
 * BestXPerfProgMode()
 *
 * Purpose: Switches performance into programming mode (DBI clock). 
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXPerfProgMode(bx_handletype handle,bx_int32 SoftForce)
{
  BX_DECLARE_FUNCNAME("BestXPerfProgMode [pprogmode]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
      /* Layout of sync-mechanism:
         1. MEPHISTO_OFFSET+0x28: 
            -32 bit control register, RW
            -only bit 0 implemented: 0=Progmode,1=Runmode
         2. MEPHISTO_OFFSET+0x2c: 
            -32 bit ready-,sync- and force register:
            -Bit 0 is sync bit from control register
            -Bit 1 is ready bit: 0=ready, 1=busy
            -Writing an arbitrary value forces bit 0 from 
             0x20 to 0x24
      */

      BX_TRY(BestXSyncRegWrite(handle,
                             
                             BX_REG_AN_CLK_SWITCH_STAT_REG,
                             sizeof(bx_int32),
                             0x2,  /* Mask for Bit 1    */
                             0x0,  /* 0 in Bit 1=Ready  */
                             
                             BX_REG_AN_CLK_SWITCH_REG,
                             sizeof(bx_int32),
                             0x1, /* Mask for bit 0 */
                             0,   /* 0=ProgMode     */
                             
                             BX_REG_AN_CLK_SWITCH_STAT_REG,
    
                             BX_REG_AN_CLK_SWITCH_STAT_REG, /* Force register */
                             sizeof(bx_int32),
                             0 /* no soft force */
            ));
                              
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXPerfRunMode()
 *
 * Purpose: Switches performance into run mode (DBI clock). 
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXPerfRunMode(bx_handletype   handle,bx_int32 SoftForce)
{
  BX_DECLARE_FUNCNAME("BestXPerfRunMode [prunmode]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
      /* Layout of sync-mechanism:
         1. MEPHISTO_OFFSET+0x28: 
            -32 bit control register, RW
            -only bit 0 implemented: 0=Progmode,1=Runmode
         2. MEPHISTO_OFFSET+0x2c: 
            -32 bit ready-,sync- and force register:
            -Bit 0 is sync bit from control register
            -Bit 1 is ready bit: 0=ready, 1=busy
            -Writing an arbitrary value forces bit 0 from 
             0x20 to 0x24
      */

      /* Switch to runmode */
      BX_TRY(BestXSyncRegWrite(handle,
                             
                             BX_REG_AN_CLK_SWITCH_STAT_REG,
                             sizeof(bx_int32),
                             0x2,  /* Mask for Bit 1    */
                             0x0,  /* 0 in Bit 1=Ready  */
                             
                             BX_REG_AN_CLK_SWITCH_REG,
                             sizeof(bx_int32),
                             0x1, /* Mask for bit 0 */
                             1,   /* 1=RunMode      */
                             
                             BX_REG_AN_CLK_SWITCH_STAT_REG,
    
                             BX_REG_AN_CLK_SWITCH_STAT_REG, /* Force register */
                             sizeof(bx_int32),
                             0 /* no soft force */
                             ));
                              
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXPerfIsInProgMode()
 *
 * Purpose: Checks, wether CAPI thinks that card is in prog-mode or not 
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXPerfIsInProgMode(bx_handletype handle,
                                             bx_int32 *ProgMode)
{
  BX_DECLARE_FUNCNAME("BestXPerfIsInProgMode [pisinprogmode]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    if (ProgMode)
    {
      BX_TRY(BestXDirectRegRead(handle,BX_REG_AN_CLK_SWITCH_STAT_REG,sizeof(bx_int32),ProgMode));
      *ProgMode=((*ProgMode)&1?0:1);
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}
